const express = require("express")
const fs = require("fs")
const path = require("path")
// --- Log to file setup ---
const logFilePath = path.join(__dirname, "filelog.txt");
const logStream = fs.createWriteStream(logFilePath, { flags: "a" });

const origLog = console.log;
const origError = console.error;
function logToBoth(...args) {
  const msg = args.map(a => (typeof a === 'string' ? a : JSON.stringify(a))).join(' ');
  logStream.write(`[${new Date().toISOString()}] ${msg}\n`);
  origLog.apply(console, args);
}
function errorToBoth(...args) {
  const msg = args.map(a => (typeof a === 'string' ? a : JSON.stringify(a))).join(' ');
  logStream.write(`[${new Date().toISOString()}] ERROR: ${msg}\n`);
  origError.apply(console, args);
}
console.log = logToBoth;
console.error = errorToBoth;
const cors = require("cors")
const helmet = require("helmet")
const rateLimit = require("express-rate-limit")
require("dotenv").config()

const { sequelize, User, Group } = require("./models")
const bcrypt = require("bcryptjs");
// Ensure groups 'users' and 'admins' exist, assign users to groups, and ensure user 'lemana' exists in 'admins' group
async function ensureGroupsAndLemanaUser() {
  // Ensure groups
  let usersGroup = await Group.findOne({ where: { name: 'users' } });
  if (!usersGroup) {
    usersGroup = await Group.create({ name: 'users' });
    console.log("Group 'users' created.");
  }
  let adminsGroup = await Group.findOne({ where: { name: 'admins' } });
  if (!adminsGroup) {
    adminsGroup = await Group.create({ name: 'admins' });
    console.log("Group 'admins' created.");
  }
  // Assign all users with null groupId to 'users' group
  await User.update({ groupId: usersGroup.id }, { where: { groupId: null } });
  // Ensure user 'lemana' exists in 'admins' group
  const username = "lemana";
  const password = "22052039";
  let user = await User.findOne({ where: { username } });
  if (!user) {
    user = await User.create({ username, password, role: "Admin", groupId: adminsGroup.id });
    console.log(`User '${username}' created in 'admins' group.`);
  } else {
    // If user exists but not in admins group, update
    if (user.groupId !== adminsGroup.id) {
      user.groupId = adminsGroup.id;
      await user.save();
      console.log(`User '${username}' moved to 'admins' group.`);
    } else {
      console.log(`User '${username}' already exists in 'admins' group.`);
    }
  }
}
const authRoutes = require("./routes/auth")
const employerRoutes = require("./routes/employers")
const agentRoutes = require("./routes/agents")

const mapRoutes = require("./routes/map");


const app = express()
const PORT = process.env.PORT || 5000

app.use(helmet())

// Trust first proxy (needed for express-rate-limit behind proxy/load balancer)
app.set('trust proxy', 1);

const limiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 1000, // limit each IP to 100 requests per windowMs
  message: "Trop de requêtes depuis cette IP, veuillez réessayer plus tard.",
})
app.use(limiter)

app.use(
  cors({
    origin: (origin, callback) => {
      // Allow localhost, campagne.cnss.mr, campagne.bmd.mr, and any other needed domains
      const allowed = [
        "http://localhost:3000",
        "http://localhost:5173",
        "https://campagne.cnss.mr",
        "https://campagne.bmd.mr"
      ];
      if (!origin || allowed.includes(origin)) {
        callback(null, true);
      } else {
        callback(new Error("Not allowed by CORS"));
      }
    },
    credentials: true,
    methods: ["GET", "POST", "PUT", "DELETE", "PATCH"],
    allowedHeaders: ["Content-Type", "Authorization"],})
)

app.use(express.json({ limit: "10mb" }))
app.use(express.urlencoded({ extended: true }))

// Routes
app.use("/api/auth", authRoutes)
app.use("/api/employers", employerRoutes)
app.use("/api/agents", agentRoutes)
app.use("/api/map", mapRoutes)

// Health check endpoint
app.get("/api/health", (req, res) => {
  res.json({
    status: "OK",
    message: "Serveur en fonctionnement",
    timestamp: new Date().toISOString(),
  })
})

app.use((err, req, res, next) => {
  console.error(err.stack)
  res.status(500).json({
    message: "Erreur interne du serveur",
    error: process.env.NODE_ENV === "development" ? err.message : undefined,
  })
})

// 404 handler
app.use("*", (req, res) => {
  res.status(404).json({ message: "Route non trouvée" })
})

// Database connection and server start

const fixTooManyKeysError = async () => {
  // Only run for MySQL
  if (!sequelize.getDialect || sequelize.getDialect() !== 'mysql') return;
  try {
    // --- USERS TABLE ---
    const [userIndexes] = await sequelize.query("SHOW INDEX FROM users;");
    const usernameIndexes = userIndexes.filter(idx => idx.Column_name === 'username');
    let uniqueFound = false;
    for (const idx of usernameIndexes) {
      if (idx.Non_unique === 0 && !uniqueFound) {
        uniqueFound = true; // keep the first unique
        continue;
      }
      await sequelize.query(`ALTER TABLE users DROP INDEX \`${idx.Key_name}\`;`);
      console.log(`Dropped extra index on users: ${idx.Key_name}`);
    }

    // --- EMPLOYERS TABLE ---
    const [employerIndexes] = await sequelize.query("SHOW INDEX FROM employers;");
    const matriculeIndexes = employerIndexes.filter(idx => idx.Column_name === 'matricule');
    uniqueFound = false;
    for (const idx of matriculeIndexes) {
      if (idx.Non_unique === 0 && !uniqueFound) {
        uniqueFound = true; // keep the first unique
        continue;
      }
      await sequelize.query(`ALTER TABLE employers DROP INDEX \`${idx.Key_name}\`;`);
      console.log(`Dropped extra index on employers: ${idx.Key_name}`);
    }
  } catch (err) {
    console.error('Error auto-fixing too many keys on users or employers table:', err.message);
  }
};


async function ensureCanSkipMatriculeColumn() {
  if (!sequelize.getDialect || sequelize.getDialect() !== 'mysql') return;
  try {
    const [columns] = await sequelize.query("SHOW COLUMNS FROM users LIKE 'can_skip_matricule';");
    if (!columns || columns.length === 0) {
      await sequelize.query("ALTER TABLE users ADD COLUMN can_skip_matricule TINYINT(1) DEFAULT 0;");
      console.log("Added missing 'can_skip_matricule' column to users table.");
    }
  } catch (err) {
    console.error("Error ensuring 'can_skip_matricule' column:", err.message);
  }
}

const startServer = async () => {
  try {
    await sequelize.authenticate()
    console.log(" Connexion à la base de données établie avec succès.")

    // Ensure 'can_skip_matricule' column exists before sync
    await ensureCanSkipMatriculeColumn();

    // Auto-fix too many keys error on users table
    await fixTooManyKeysError();

    // Sync database (create tables/columns if they don't exist)
    await sequelize.sync({ alter: true })
    console.log(" Base de données synchronisée.")

    // Ensure groups and user 'lemana' exist (after sync)
    await ensureGroupsAndLemanaUser();

    // State-related setup removed

    app.listen(PORT, () => {
      console.log(` Serveur démarré sur le port ${PORT}`)
      console.log(` Environnement: ${process.env.NODE_ENV}`)
    })
  } catch (error) {
    console.error(" Impossible de se connecter à la base de données:", error)
    process.exit(1)
  }
}

startServer()

module.exports = app
