const { DataTypes } = require("sequelize")
const bcrypt = require("bcryptjs")
const sequelize = require("../config/database")

const User = sequelize.define(
  "User",
  {
    id: {
      type: DataTypes.INTEGER,
      primaryKey: true,
      autoIncrement: true,
    },
    username: {
      type: DataTypes.STRING(50),
      allowNull: false,
      unique: true,
      validate: {
        len: [3, 50],
        notEmpty: true,
      },
    },
    password: {
      type: DataTypes.STRING(255),
      allowNull: false,
      validate: {
        len: [6, 255],
        notEmpty: true,
      },
    },
    role: {
      type: DataTypes.ENUM("Admin", "Agent"),
      allowNull: false,
      defaultValue: "Agent",
    },
    state: {
      type: DataTypes.ENUM(
        "Adrar",
        "Assaba",
        "Brakna",
        "Dakhlet Nouadhibou",
        "Gorgol",
        "Guidimakha",
        "Hodh Ech Chargui",
        "Hodh El Gharbi",
        "Inchiri",
        "Nouakchott Nord",
        "Nouakchott Ouest",
        "Nouakchott Sud",
        "Tagant",
        "Tiris Zemmour",
        "Trarza"
      ),
      allowNull: false,
      defaultValue: "Nouakchott Nord",
    },
    can_skip_matricule: {
      type: DataTypes.BOOLEAN,
      allowNull: false,
      defaultValue: false,
    },
    groupId: {
      type: DataTypes.INTEGER,
      allowNull: true, // will be set after groups exist
      references: {
        model: 'groups',
        key: 'id',
      },
    },
  },
  {
    tableName: "users",
    hooks: {
      beforeCreate: async (user) => {
        if (user.password) {
          user.password = await bcrypt.hash(user.password, 12)
        }
      },
      beforeUpdate: async (user) => {
        if (user.changed("password")) {
          user.password = await bcrypt.hash(user.password, 12)
        }
      },
    },
  },
)

User.prototype.checkPassword = async function (password) {
  return await bcrypt.compare(password, this.password)
}

User.findByUsername = function (username) {
  return this.findOne({ where: { username } })
}

module.exports = User
