const { validationResult } = require("express-validator");
const { Employer, User } = require("../models");
const { Op } = require("sequelize");

// Get all employers
const getAllEmployers = async (req, res) => {
  try {
    let employers;
    if (req.user.role === "Admin") {
      employers = await Employer.findAll({
        include: [
          {
            model: User,
            as: "agent",
            attributes: ["id", "username"],
            include: [{ model: require("../models").Group, as: "group", attributes: ["id", "name"] }],
          },
          { model: User, as: "deletedByUser", attributes: ["id", "username"] },
        ],
        order: [["created_at", "DESC"]],
      });
    } else {
      employers = await Employer.findAll({
        where: { agent_id: req.user.id, deleted: false },
        include: [
          {
            model: User,
            as: "agent",
            attributes: ["id", "username"],
            include: [{ model: require("../models").Group, as: "group", attributes: ["id", "name"] }],
          },
          { model: User, as: "deletedByUser", attributes: ["id", "username"] },
        ],
        order: [["created_at", "DESC"]],
      });
    }
    const employersWithIsoDates = employers.map((e) => {
      const obj = e.toJSON();
      if (obj.created_at) obj.created_at = new Date(obj.created_at).toISOString();
      if (obj.updated_at) obj.updated_at = new Date(obj.updated_at).toISOString();
      return obj;
    });
    res.json(employersWithIsoDates);
  } catch (error) {
    console.error("Erreur lors de la récupération des employeurs:", error);
    res.status(500).json({ message: "Erreur interne du serveur" });
  }
};

// Get employer by ID
const getEmployerById = async (req, res) => {
  try {
    const employer = await Employer.findByPk(req.params.id, {
      include: [
        {
          model: User,
          as: "agent",
          attributes: ["id", "username"],
          include: [{ model: require("../models").Group, as: "group", attributes: ["id", "name"] }],
        },
        { model: User, as: "deletedByUser", attributes: ["id", "username"] },
      ],
    });
    if (!employer) return res.status(404).json({ message: "Employeur non trouvé" });

    if (req.user.role !== "Admin" && employer.agent_id !== req.user.id) {
      return res.status(403).json({ message: "Accès non autorisé" });
    }
    const obj = employer.toJSON();
    if (obj.created_at) obj.created_at = new Date(obj.created_at).toISOString();
    if (obj.updated_at) obj.updated_at = new Date(obj.updated_at).toISOString();
    res.json(obj);
  } catch (error) {
    console.error("Erreur lors de la récupération de l'employeur:", error);
    res.status(500).json({ message: "Erreur interne du serveur" });
  }
};

// Create new employer
const createEmployer = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty())
      return res.status(400).json({ message: "Données invalides", errors: errors.array() });

    const user = await User.findByPk(req.user.id);
    const { matricule, commerce_register, ...otherFields } = req.body;

    let finalMatricule = matricule;
    if (user.can_skip_matricule && (!matricule || matricule === "-")) {
      finalMatricule = null;
    }

    if (!user.can_skip_matricule && (!matricule || matricule.trim() === "")) {
      return res.status(400).json({ message: "Le matricule est requis." });
    }

    if (finalMatricule !== null) {
      const existingEmployer = await Employer.findOne({ where: { matricule: finalMatricule } });
      if (existingEmployer) {
        return res.status(400).json({ message: "Ce matricule existe déjà" });
      }
    }

    const employer = await Employer.create({
      matricule: finalMatricule,
      commerce_register: commerce_register || null,
      ...otherFields,
      agent_id: user.id,
    });

    res.status(201).json(employer);
  } catch (error) {
    console.error("Erreur lors de la création de l'employeur:", error);
    res.status(500).json({ message: "Erreur interne du serveur" });
  }
};

// Update employer
const updateEmployer = async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty())
      return res.status(400).json({ message: "Données invalides", errors: errors.array() });

    const employer = await Employer.findByPk(req.params.id);
    if (!employer) return res.status(404).json({ message: "Employeur non trouvé" });

    if (req.user.role !== "Admin" && employer.agent_id !== req.user.id) {
      return res.status(403).json({ message: "Accès non autorisé" });
    }

    const {
      matricule,
      commerce_register,
      nom_responsable,
      numero_telephone,
      latitude,
      longitude,
      employeur_nom,
      employeur_type,
      ville,
      secteur,
    } = req.body;

    if (matricule && matricule.trim() !== "") {
      const existingEmployer = await Employer.findOne({
        where: {
          matricule,
          id: { [Op.ne]: req.params.id },
        },
      });
      if (existingEmployer) {
        return res.status(400).json({ message: "Ce matricule existe déjà" });
      }
    }

    await employer.update({
      matricule,
      commerce_register: commerce_register || null,
      nom_responsable,
      numero_telephone,
      latitude: Number.parseFloat(latitude),
      longitude: Number.parseFloat(longitude),
      employeur_nom: employeur_nom || null,
      employeur_type: employeur_type || null,
      ville: ville || "Nouakchott",
      secteur,
    });

    const updatedEmployer = await Employer.findByPk(employer.id, {
      include: [{ model: User, as: "agent", attributes: ["id", "username"] }],
    });

    res.json({ message: "Employeur mis à jour avec succès", employer: updatedEmployer });
  } catch (error) {
    console.error("Erreur lors de la mise à jour de l'employeur:", error);
    res.status(500).json({ message: "Erreur interne du serveur" });
  }
};

// Soft delete employer
const softDeleteEmployer = async (req, res) => {
  try {
    const employer = await Employer.findByPk(req.params.id);
    if (!employer) return res.status(404).json({ message: "Employeur non trouvé" });

    if (req.user.role !== "Admin" && employer.agent_id !== req.user.id) {
      return res.status(403).json({ message: "Accès non autorisé" });
    }

    employer.deleted = true;
    employer.deleted_by = req.user.id;
    await employer.save();

    res.json({ message: "Employeur masqué (supprimé) avec succès" });
  } catch (error) {
    console.error("Erreur lors de la suppression de l'employeur:", error);
    res.status(500).json({ message: "Erreur interne du serveur" });
  }
};

// Undelete employer (admin only)
const undeleteEmployer = async (req, res) => {
  try {
    if (req.user.role !== "Admin") {
      return res.status(403).json({ message: "Seul l'administrateur peut restaurer un employeur." });
    }
    const employer = await Employer.findByPk(req.params.id);
    if (!employer) return res.status(404).json({ message: "Employeur non trouvé" });

    employer.deleted = false;
    employer.deleted_by = null;
    await employer.save();

    res.json({ message: "Employeur restauré avec succès" });
  } catch (error) {
    console.error("Erreur lors de la restauration de l'employeur:", error);
    res.status(500).json({ message: "Erreur interne du serveur" });
  }
};

// Permanent delete employer (admin only)
const permanentDeleteEmployer = async (req, res) => {
  try {
    if (req.user.role !== "Admin") {
      return res.status(403).json({ message: "Accès refusé." });
    }

    const employer = await Employer.findByPk(req.params.id);
    if (!employer) return res.status(404).json({ message: "Employeur non trouvé" });

    if (!employer.deleted) {
      return res
        .status(400)
        .json({ message: "L'employeur doit être d'abord supprimé (soft delete) avant suppression définitive." });
    }

    await employer.destroy();
    res.json({ message: "Employeur supprimé définitivement avec succès" });
  } catch (error) {
    console.error("Erreur lors de la suppression définitive de l'employeur:", error);
    res.status(500).json({ message: "Erreur interne du serveur" });
  }
};

module.exports = {
  getAllEmployers,
  getEmployerById,
  createEmployer,
  updateEmployer,
  softDeleteEmployer,
  undeleteEmployer,
  permanentDeleteEmployer,
};
