const { Employeur, User } = require("../models")
const ExcelJS = require("exceljs")
const PDFDocument = require("pdfkit")
const {
  InternalServerError,
  enumErrorMessages,
} = require("../errors")

const getSummary = async (req, res) => {
  try {
    const isAdmin = req.user.role === "Admin"
    let employeurs, agents

    if (isAdmin) {
      employeurs = await Employeur.findAll({
        include: [{ model: User, as: "agent", attributes: ["id", "username"] }],
      })
      agents = await User.findAll({
        where: { role: "Agent" },
        attributes: ["id", "username"],
      })
    } else {
      employeurs = await Employeur.findAll({
        where: { agent_id: req.user.id },
        include: [{ model: User, as: "agent", attributes: ["id", "username"] }],
      })
      agents = [{ id: req.user.id, username: req.user.username }]
    }

    const totalEmployeurs = employeurs.length
    const totalAgents = agents.length

    const employeursPerAgent = agents.map((agent) => ({
      agent_id: agent.id,
      username: agent.username,
      count: employeurs.filter((e) => e.agent_id === agent.id).length,
    }))

    const byCity = {}
    employeurs.forEach((e) => {
      const city = e.ville || "Inconnu"
      byCity[city] = (byCity[city] || 0) + 1
    })

    res.json({
      employeurs,
      stats: {
        totalEmployeurs,
        totalAgents,
        employeursPerAgent,
        byCity,
      },
    })
  } catch (err) {
    console.error("Dashboard error:", err)
    res
      .status(500)
      .json(InternalServerError(enumErrorMessages.DASHBOARD_FETCH_ERROR))
  }
}

const exportExcel = async (req, res) => {
  try {
    const isAdmin = req.user.role === "Admin"
    let employeurs

    if (isAdmin) {
      employeurs = await Employeur.findAll({
        include: [{ model: User, as: "agent", attributes: ["username"] }],
      })
    } else {
      employeurs = await Employeur.findAll({
        where: { agent_id: req.user.id },
        include: [{ model: User, as: "agent", attributes: ["username"] }],
      })
    }

    const workbook = new ExcelJS.Workbook()
    const sheet = workbook.addWorksheet("Employeurs")

    sheet.columns = [
      { header: "Matricule", key: "matricule" },
      { header: "Nom", key: "employeur_nom" },
      { header: "Responsable", key: "nom_responsable" },
      { header: "Téléphone", key: "numero_telephone" },
      { header: "Ville", key: "ville" },
      { header: "Type", key: "employeur_type" },
      { header: "Secteur", key: "secteur" },
      { header: "Latitude", key: "latitude" },
      { header: "Longitude", key: "longitude" },
      { header: "Agent", key: "agent" },
    ]

    employeurs.forEach((e) => {
      sheet.addRow({ ...e.dataValues, agent: e.agent?.username })
    })

    res.setHeader(
      "Content-Type",
      "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"
    )
    res.setHeader(
      "Content-Disposition",
      'attachment; filename="employeurs.xlsx"'
    )

    await workbook.xlsx.write(res)
    res.end()
  } catch (err) {
    console.error("Excel export error:", err)
    res
      .status(500)
      .json(InternalServerError(enumErrorMessages.EXCEL_EXPORT_ERROR))
  }
}

const exportPdf = async (req, res) => {
  try {
    const isAdmin = req.user.role === "Admin"
    let employeurs

    if (isAdmin) {
      employeurs = await Employeur.findAll({
        include: [{ model: User, as: "agent", attributes: ["username"] }],
      })
    } else {
      employeurs = await Employeur.findAll({
        where: { agent_id: req.user.id },
        include: [{ model: User, as: "agent", attributes: ["username"] }],
      })
    }

    const doc = new PDFDocument()
    res.setHeader("Content-Type", "application/pdf")
    res.setHeader("Content-Disposition", 'attachment; filename="employeurs.pdf"')

    doc.pipe(res)
    doc.fontSize(16).text("Liste des Employeurs", { align: "center" })
    doc.moveDown()

    employeurs.forEach((e) => {
      doc
        .fontSize(10)
        .text(
          `Matricule: ${e.matricule} | Nom: ${e.employeur_nom} | Responsable: ${e.nom_responsable} | Téléphone: ${e.numero_telephone} | Ville: ${e.ville} | Type: ${e.employeur_type} | Secteur: ${e.secteur} | Agent: ${e.agent?.username}`
        )
      doc.moveDown(0.5)
    })

    doc.end()
  } catch (err) {
    console.error("PDF export error:", err)
    res
      .status(500)
      .json(InternalServerError(enumErrorMessages.PDF_EXPORT_ERROR))
  }
}

module.exports = {
  getSummary,
  exportExcel,
  exportPdf,
}
