const jwt = require("jsonwebtoken")
const { validationResult } = require("express-validator")
const { User } = require("../models")

const {
  BadRequestError,
  UnauthorizedError,
  InternalServerError,
  enumErrorMessages,
} = require("../errors")

const login = async (req, res) => {
  try {
    const errors = validationResult(req)
    if (!errors.isEmpty()) {
      return res
        .status(400)
        .json(BadRequestError(enumErrorMessages.VALIDATION_FAILED, errors.array()))
    }

    const { username, password } = req.body
    const user = await User.findByUsername(username)

    if (!user || !(await user.checkPassword(password))) {
      return res
        .status(401)
        .json(UnauthorizedError(enumErrorMessages.INVALID_CREDENTIALS))
    }

    const token = jwt.sign(
      {
        id: user.id,
        username: user.username,
        role: user.role,
      },
      process.env.JWT_SECRET,
      { expiresIn: process.env.JWT_EXPIRES_IN || "24h" }
    )

    res.json({
      message: enumErrorMessages.LOGIN_SUCCESS, // custom success message (see below)
      token,
      user: {
        id: user.id,
        username: user.username,
        role: user.role,
        can_skip_matricule: user.can_skip_matricule,
      },
    })
  } catch (error) {
    console.error("Erreur de connexion:", error)
    res
      .status(500)
      .json(InternalServerError(enumErrorMessages.INTERNAL_SERVER_ERROR))
  }
}

const verify = (req, res) => {
  res.json({
    user: {
      id: req.user.id,
      username: req.user.username,
      role: req.user.role,
      can_skip_matricule: req.user.can_skip_matricule,
    },
  })
}

const logout = (req, res) => {
  res.json({ message: enumErrorMessages.LOGOUT_SUCCESS })
}

module.exports = {
  login,
  verify,
  logout,
}
