const { validationResult } = require("express-validator")
const { User, Group } = require("../models")
const { Op } = require("sequelize")

const getGroups = async (req, res) => {
  try {
    const groups = await Group.findAll({ attributes: ["id", "name", "created_at", "updated_at"] })
    res.json(groups)
  } catch (error) {
    console.error("Erreur lors de la récupération des groupes:", error)
    res.status(500).json({ message: "Erreur interne du serveur" })
  }
}

const createGroup = async (req, res) => {
  const errors = validationResult(req)
  if (!errors.isEmpty()) {
    return res.status(400).json({ message: "Données invalides", errors: errors.array() })
  }

  try {
    const { name } = req.body
    const existing = await Group.findOne({ where: { name } })
    if (existing) return res.status(400).json({ message: "Ce groupe existe déjà" })

    const group = await Group.create({ name })
    res.status(201).json({ message: "Groupe créé avec succès", group })
  } catch (error) {
    console.error("Erreur lors de la création du groupe:", error)
    res.status(500).json({ message: "Erreur interne du serveur" })
  }
}

const updateGroup = async (req, res) => {
  const errors = validationResult(req)
  if (!errors.isEmpty()) return res.status(400).json({ message: "Données invalides", errors: errors.array() })

  try {
    const group = await Group.findByPk(req.params.id)
    if (!group) return res.status(404).json({ message: "Groupe non trouvé" })

    group.name = req.body.name
    await group.save()
    res.json({ message: "Groupe mis à jour", group })
  } catch (error) {
    console.error("Erreur lors de la mise à jour du groupe:", error)
    res.status(500).json({ message: "Erreur interne du serveur" })
  }
}

const deleteGroup = async (req, res) => {
  try {
    const group = await Group.findByPk(req.params.id)
    if (!group) return res.status(404).json({ message: "Groupe non trouvé" })
    if (["users", "admins"].includes(group.name)) return res.status(400).json({ message: "Impossible de supprimer ce groupe système" })

    await group.destroy()
    res.json({ message: "Groupe supprimé avec succès" })
  } catch (error) {
    console.error("Erreur lors de la suppression du groupe:", error)
    res.status(500).json({ message: "Erreur interne du serveur" })
  }
}

// --- AGENTS ---
const listAgents = async (req, res) => {
  try {
    const agents = await User.findAll({
      attributes: ["id", "username", "role", "state", "created_at", "updated_at", "can_skip_matricule", "groupId"],
      include: [{ model: Group, as: "group", attributes: ["id", "name"] }],
      order: [["created_at", "DESC"]],
    })
    res.json(agents)
  } catch (error) {
    console.error("Erreur lors de la récupération des agents:", error)
    res.status(500).json({ message: "Erreur interne du serveur" })
  }
}

const getAgentById = async (req, res) => {
  try {
    const agent = await User.findByPk(req.params.id, {
      attributes: ["id", "username", "role", "state", "created_at", "updated_at", "can_skip_matricule", "groupId"],
      include: [{ model: Group, as: "group", attributes: ["id", "name"] }],
    })
    if (!agent) return res.status(404).json({ message: "Agent non trouvé" })
    res.json(agent)
  } catch (error) {
    console.error("Erreur lors de la récupération de l'agent:", error)
    res.status(500).json({ message: "Erreur interne du serveur" })
  }
}

const createAgent = async (req, res) => {
  const errors = validationResult(req)
  if (!errors.isEmpty()) return res.status(400).json({ message: "Données invalides", errors: errors.array() })

  try {
    const { username, password, role, state, can_skip_matricule, groupId } = req.body

    const existingUser = await User.findOne({ where: { username } })
    if (existingUser) return res.status(400).json({ message: "Ce nom d'utilisateur existe déjà" })

    let finalGroupId = groupId
    if (!finalGroupId) {
      const usersGroup = await Group.findOne({ where: { name: "users" } })
      finalGroupId = usersGroup ? usersGroup.id : null
    }

    const agentData = { username, password, role, state, groupId: finalGroupId }
    if (typeof can_skip_matricule === "boolean") {
      agentData.can_skip_matricule = can_skip_matricule
    }

    const agent = await User.create(agentData)

    const agentWithGroup = await User.findByPk(agent.id, {
      attributes: ["id", "username", "role", "state", "created_at", "can_skip_matricule", "groupId"],
      include: [{ model: Group, as: "group", attributes: ["id", "name"] }],
    })

    res.status(201).json({ message: "Agent créé avec succès", agent: agentWithGroup })
  } catch (error) {
    console.error("Erreur lors de la création de l'agent:", error)
    res.status(500).json({ message: "Erreur interne du serveur" })
  }
}

const updateAgent = async (req, res) => {
  const errors = validationResult(req)
  if (!errors.isEmpty()) return res.status(400).json({ message: "Données invalides", errors: errors.array() })

  try {
    const agent = await User.findByPk(req.params.id)
    if (!agent) return res.status(404).json({ message: "Agent non trouvé" })

    const { username, password, role, state, can_skip_matricule, groupId } = req.body

    const existingUser = await User.findOne({
      where: { username, id: { [Op.ne]: req.params.id } },
    })
    if (existingUser) return res.status(400).json({ message: "Ce nom d'utilisateur existe déjà" })

    const updateData = { username, role, state }
    if (password) updateData.password = password
    if (typeof can_skip_matricule === "boolean") updateData.can_skip_matricule = can_skip_matricule
    if (groupId) updateData.groupId = groupId

    await agent.update(updateData)

    const updated = await User.findByPk(agent.id, {
      attributes: ["id", "username", "role", "state", "updated_at", "can_skip_matricule", "groupId"],
      include: [{ model: Group, as: "group", attributes: ["id", "name"] }],
    })

    res.json({ message: "Agent mis à jour avec succès", agent: updated })
  } catch (error) {
    console.error("Erreur lors de la mise à jour de l'agent:", error)
    res.status(500).json({ message: "Erreur interne du serveur" })
  }
}

const deleteAgent = async (req, res) => {
  try {
    const agent = await User.findByPk(req.params.id)
    if (!agent) return res.status(404).json({ message: "Agent non trouvé" })
    if (agent.id === req.user.id) return res.status(400).json({ message: "Vous ne pouvez pas supprimer votre propre compte" })

    await agent.destroy()
    res.json({ message: "Agent supprimé avec succès" })
  } catch (error) {
    console.error("Erreur lors de la suppression de l'agent:", error)
    res.status(500).json({ message: "Erreur interne du serveur" })
  }
}

const patchCanSkip = async (req, res) => {
  try {
    const agent = await User.findByPk(req.params.id)
    if (!agent) return res.status(404).json({ message: "Agent non trouvé" })

    if (typeof req.body.can_skip_matricule !== "boolean") {
      return res.status(400).json({ message: "Valeur can_skip_matricule invalide" })
    }

    agent.can_skip_matricule = req.body.can_skip_matricule
    await agent.save()

    res.json({ message: "Mise à jour réussie", can_skip_matricule: agent.can_skip_matricule })
  } catch (error) {
    console.error("Erreur lors de la mise à jour de can_skip_matricule:", error)
    res.status(500).json({ message: "Erreur interne du serveur" })
  }
}

module.exports = {
  getGroups,
  createGroup,
  updateGroup,
  deleteGroup,
  listAgents,
  getAgentById,
  createAgent,
  updateAgent,
  deleteAgent,
  patchCanSkip,
}
