const { body } = require("express-validator")

const validateLogin = [
  body("username")
    .trim()
    .isLength({ min: 3 })
    .withMessage("Le nom d'utilisateur doit contenir au moins 3 caractères"),

  body("password")
    .isLength({ min: 6 })
    .withMessage("Le mot de passe doit contenir au moins 6 caractères"),
]

const validateAgent = [
  body("username").trim().isLength({ min: 3, max: 50 }).withMessage("Le nom d'utilisateur doit contenir entre 3 et 50 caractères"),
  body("password").optional().isLength({ min: 6 }).withMessage("Le mot de passe doit contenir au moins 6 caractères"),
  body("role").isIn(["Admin", "Agent"]).withMessage("Le rôle doit être Admin ou Agent"),
  body("state").isIn([
    "Adrar", "Assaba", "Brakna", "Dakhlet Nouadhibou", "Gorgol", "Guidimakha",
    "Hodh Ech Chargui", "Hodh El Gharbi", "Inchiri", "Nouakchott Nord", "Nouakchott Ouest",
    "Nouakchott Sud", "Tagant", "Tiris Zemmour", "Trarza"
  ]).withMessage("La région sélectionnée est invalide"),
]

const validateGroup = [
  body("name").trim().isLength({ min: 2, max: 50 }).withMessage("Le nom du groupe doit contenir entre 2 et 50 caractères"),
]

const validateCreateAgent = [...validateAgent, body("password").notEmpty().withMessage("Le mot de passe est requis pour créer un agent")]

const EMPLOYEUR_TYPES = [
  "Banque",
  "Pharmacie",
  "Super Marcher",
  "Station Service",
  "Restaurant",
  "Hôtel",
  "Clinique",
  "Bureau de Change",
  "Assurance",
  "Transport",
  "École",
  "Université",
  "Autre"
];
const MAURITANIAN_REGIONS = [
  "Nouakchott",
  "Adrar",
  "Assaba",
  "Brakna",
  "Dakhlet Nouadhibou",
  "Gorgol",
  "Guidimakha",
  "Hodh Ech Chargui",
  "Hodh El Gharbi",
  "Inchiri",
  "Nouakchott Nord",
  "Nouakchott Ouest",
  "Nouakchott Sud",
  "Tagant",
  "Tiris Zemmour",
  "Trarza"
];

// Custom matricule validator based on user.can_skip_matricule
const validateMatricule = body("matricule").custom((value, { req }) => {
  if (req.user && req.user.can_skip_matricule) {
    if (value && value.length > 50) {
      throw new Error("Le matricule ne peut pas dépasser 50 caractères");
    }
    return true;
  } else {
    if (!value || value.trim() === "") {
      throw new Error("Le matricule est requis");
    }
    if (value.length > 50) {
      throw new Error("Le matricule ne peut pas dépasser 50 caractères");
    }
    return true;
  }
});

const validateEmployer = [
  validateMatricule,
  body("nom_responsable")
    .trim()
    .notEmpty()
    .withMessage("Le nom du responsable est requis")
    .isLength({ max: 100 })
    .withMessage("Le nom ne peut pas dépasser 100 caractères"),
  body("numero_telephone")
    .trim()
    .notEmpty()
    .withMessage("Le numéro de téléphone est requis")
    .isLength({ max: 20 })
    .withMessage("Le numéro de téléphone ne peut pas dépasser 20 caractères"),
  body("latitude")
    .isFloat({ min: -90, max: 90 })
    .withMessage("La latitude doit être entre -90 et 90"),
  body("longitude")
    .isFloat({ min: -180, max: 180 })
    .withMessage("La longitude doit être entre -180 et 180"),
  body("employeur_nom")
    .optional()
    .isLength({ max: 100 }),
  body("commerce_register")
    .optional()
    .isLength({ max: 100 }),
  body("employeur_type")
    .optional()
    .isIn(EMPLOYEUR_TYPES),
  body("ville")
    .optional()
    .isIn(MAURITANIAN_REGIONS)
    .withMessage("La région sélectionnée est invalide"),
  body("secteur")
    .notEmpty()
    .withMessage("Le secteur est requis")
    .isIn(["Public", "Privé"])
    .withMessage("Le secteur doit être 'Public' ou 'Privé'"),
];
module.exports = {
  validateLogin,
  validateAgent,
  validateGroup,
  validateCreateAgent,
  validateEmployer
}
